// Main editor functionality

// Generate and download page
function generateAndDownload(config) {
  try {
    console.log('Generating and downloading page');
    const template = generatePageTemplate(config);
    
    // Use page title as filename
    let fileName = config.pageTitle1 || 'untitled';
    // Sanitize invalid filename characters
    fileName = fileName.replace(/[<>:"/\\|?*]+/g, '-');
    fileName = fileName.trim();
    // Use default name if filename is empty
    if (!fileName) {
      fileName = 'page';
    }
    fileName += '.html';
    
    // Create Blob and trigger download
    const blob = new Blob([template], { type: 'text/html' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = fileName;
    document.body.appendChild(a);
    a.click();
    window.URL.revokeObjectURL(url);
    document.body.removeChild(a);
    
const msgDiv = document.createElement('div');
msgDiv.style.position = 'fixed';
msgDiv.style.bottom = '20px';
msgDiv.style.right = '20px';
msgDiv.style.padding = '15px 20px';
msgDiv.style.backgroundColor = 'rgba(0, 128, 0, 0.9)';
msgDiv.style.color = 'white';
msgDiv.style.borderRadius = '10px';
msgDiv.style.zIndex = '9999';
msgDiv.style.boxShadow = '0 4px 12px rgba(0, 0, 0, 0.15)';
msgDiv.style.animation = 'fadeIn 0.5s ease';
msgDiv.innerHTML = `
  <h3 style="margin-top: 0">Page Generated Successfully!</h3>
  <p>Please create the following folder structure:</p>
  <ul style="margin-top: 5px; padding-left: 20px;">
    <li>Place the HTML file <b>${fileName}</b> in your project root directory</li>
    <li>Create a <strong>css/</strong> folder with these files:
      <ul style="padding-left: 15px">
        <li>main.css</li>
        <li>light.css</li>
        <li>dark.css</li>
      </ul>
    </li>
    <li>Create an <strong>img/</strong> folder for all images</li>
  </ul>
  <p style="margin-top: 8px">Basic styles will display even without CSS files</p>
  <p style="margin-top: 10px; font-size: 0.8em; text-align: center;">Click this message to dismiss</p>
`;

document.body.appendChild(msgDiv);

// Make notification clickable to dismiss
msgDiv.addEventListener('click', function() {
  msgDiv.style.opacity = '0';
  msgDiv.style.transition = 'opacity 0.5s ease';
  setTimeout(() => document.body.removeChild(msgDiv), 500);
});

// Auto-dismiss after 10 seconds
setTimeout(() => {
  msgDiv.style.opacity = '0';
  msgDiv.style.transition = 'opacity 1s ease';
  setTimeout(() => {
    if (document.body.contains(msgDiv)) {
      document.body.removeChild(msgDiv);
    }
  }, 1000);
}, 10000);
  } catch (error) {
    console.error('Failed to generate page:', error);
    alert('Failed to generate page: ' + error.message);
  }
}

// Preview generation function
function generatePreview(config) {
  try {
    console.log('Generating preview');
    const previewContainer = document.getElementById('previewContainer');
    
    // Create iframe preview
    const iframe = document.createElement('iframe');
    iframe.style.width = '100%';
    iframe.style.height = '600px';
    iframe.style.border = 'none';
    
    // Clear preview container and append iframe
    previewContainer.innerHTML = '';
    previewContainer.appendChild(iframe);
    
    // Set iframe content
    const template = generatePageTemplate(config);
    iframe.srcdoc = template;
  } catch (error) {
    console.error('Preview generation failed:', error);
    
    // Display error message
    const errorDiv = document.createElement('div');
    errorDiv.className = 'error-message';
    errorDiv.textContent = `Preview generation failed: ${error.message}`;
    document.getElementById('previewContainer').innerHTML = '';
    document.getElementById('previewContainer').appendChild(errorDiv);
  }
}

// Auto preview update function
function updatePreview() {
  try {
    const config = collectFormData();
    generatePreview(config);
  } catch (error) {
    console.error('Preview update failed:', error);
  }
}

// Initialize event listeners
document.addEventListener('DOMContentLoaded', function() {
  // Theme initialization handled by theme-switcher.js
  // Initialize background image URLs
  const currentTheme = document.documentElement.getAttribute('data-theme') || 'light';
  const bgImg = currentTheme === 'dark' 
      ? document.getElementById('darkBg').value || 'img/dark-bg.jpg'
      : document.getElementById('lightBg').value || 'img/light-bg.jpg';
  
  document.documentElement.style.setProperty(`--${currentTheme}-bg-url`, `url('${bgImg}')`);
  
  // Add realtime preview
  addRealtimePreview();
  
  // Bind button events
  document.getElementById('generateBtn').addEventListener('click', function() {
    generateAndDownload(collectFormData());
  });
  
  document.getElementById('previewBtn').addEventListener('click', function() {
    generatePreview(collectFormData());
  });
  
  // Initialize some examples
  addNavMenuItem();
  addSection();
  addLinkItem('linkList1');
  
  // Initialize preview
  updatePreview();
});