// File handling related functions

// File selection handler
function handleFileSelect(input, targetId) {
  const file = input.files[0];
  if (file) {
    // Create relative path
    const fileName = file.name;
    const imgPath = `img/${fileName}`;
    
    console.log(`Processing file: ${fileName}, Target ID: ${targetId}, Path: ${imgPath}`);
    
    // Set target input value to image path
    document.getElementById(targetId).value = imgPath;
    
    // Add thumbnail preview (editor only, does not affect final HTML)
    if (file.type.startsWith('image/')) {
      const previewId = targetId + 'Preview';
      let previewElement = document.getElementById(previewId);
      
      // Create preview element if not exists
      if (!previewElement) {
        previewElement = document.createElement('img');
        previewElement.id = previewId;
        previewElement.className = 'img-preview';
        
        const inputContainer = document.getElementById(targetId).parentElement;
        inputContainer.appendChild(previewElement);
      }
      
      // Display temporary preview (editor only)
      const reader = new FileReader();
      reader.onload = function(e) {
        previewElement.src = e.target.result;
        previewElement.title = fileName;
      };
      reader.readAsDataURL(file);
    }
    
    // Trigger preview update
    setTimeout(updatePreview, 100);
  }
}

// 内容块图片处理程序，支持图片尺寸控制
function handleSectionImage(input, sectionElement) {
  const file = input.files[0];
  if (!file) return;
  
  // Create relative path
  const fileName = file.name;
  const imgPath = `img/${fileName}`;
  
  console.log(`Processing section image: ${fileName}, Path: ${imgPath}`);
  
  // 获取textarea
  const contentEditor = sectionElement.querySelector('.content-editor');
  const textarea = contentEditor.querySelector('.section-content');
  
  // 创建临时预览
  const tempPreview = document.createElement('img');
  const reader = new FileReader();
  reader.onload = function(e) {
    tempPreview.src = e.target.result;
    
    // 图片加载完成后，获取原始尺寸
    tempPreview.onload = function() {
      const origWidth = tempPreview.naturalWidth;
      const origHeight = tempPreview.naturalHeight;
      
      // 创建图片尺寸控制对话框
      const sizingDialog = document.createElement('div');
      sizingDialog.className = 'image-size-dialog';
      sizingDialog.innerHTML = `
        <h4>Set Image Size</h4>
        <div class="image-size-inputs">
          <label>Width: <input type="number" class="width-input" value="${origWidth}" min="10" max="1200"></label>
          <label>Height: <input type="number" class="height-input" value="${origHeight}" min="10" max="1200"></label>
          <label><input type="checkbox" class="keep-ratio" checked> Maintain Aspect Ratio</label>
        </div>
        <div class="size-presets">
          <button type="button" class="size-preset-btn" data-width="100" data-unit="%">100% Width</button>
          <button type="button" class="size-preset-btn" data-width="75" data-unit="%">75% Width</button>
          <button type="button" class="size-preset-btn" data-width="50" data-unit="%">50% Width</button>
          <button type="button" class="size-preset-btn" data-width="25" data-unit="%">25% Width</button>
          <button type="button" class="size-preset-btn" data-width="300" data-unit="px">Small</button>
          <button type="button" class="size-preset-btn" data-width="600" data-unit="px">Medium</button>
          <button type="button" class="size-preset-btn" data-width="900" data-unit="px">Large</button>
        </div>
        <div class="image-align-options">
          <span class="image-align-option active" data-align="none">Default</span>
          <span class="image-align-option" data-align="left">Left</span>
          <span class="image-align-option" data-align="center">Center</span>
          <span class="image-align-option" data-align="right">Right</span>
        </div>
        <div>
          <button type="button" class="btn btn-primary insert-image-btn">Insert Image</button>
          <button type="button" class="btn btn-secondary cancel-btn">Cancel</button>
        </div>
      `;
      
      // 添加对话框到编辑器
      contentEditor.appendChild(sizingDialog);
      
      // 处理图片尺寸输入
      const widthInput = sizingDialog.querySelector('.width-input');
      const heightInput = sizingDialog.querySelector('.height-input');
      const keepRatioCheckbox = sizingDialog.querySelector('.keep-ratio');
      const aspectRatio = origWidth / origHeight;
      
      // 宽度变化时调整高度
      widthInput.addEventListener('input', function() {
        if (keepRatioCheckbox.checked) {
          heightInput.value = Math.round(widthInput.value / aspectRatio);
        }
      });
      
      // 高度变化时调整宽度
      heightInput.addEventListener('input', function() {
        if (keepRatioCheckbox.checked) {
          widthInput.value = Math.round(heightInput.value * aspectRatio);
        }
      });
      
      // 处理尺寸预设按钮
      sizingDialog.querySelectorAll('.size-preset-btn').forEach(btn => {
        btn.addEventListener('click', function() {
          const presetWidth = this.getAttribute('data-width');
          const unit = this.getAttribute('data-unit');
          
          if (unit === '%') {
            // 百分比尺寸预设
            widthInput.value = Math.round(origWidth * (presetWidth / 100));
            if (keepRatioCheckbox.checked) {
              heightInput.value = Math.round(origHeight * (presetWidth / 100));
            }
          } else {
            // 固定像素尺寸预设
            widthInput.value = presetWidth;
            if (keepRatioCheckbox.checked) {
              heightInput.value = Math.round(presetWidth / aspectRatio);
            }
          }
        });
      });
      
      // 处理对齐选项
      sizingDialog.querySelectorAll('.image-align-option').forEach(option => {
        option.addEventListener('click', function() {
          // 移除所有的活动状态
          sizingDialog.querySelectorAll('.image-align-option').forEach(opt => {
            opt.classList.remove('active');
          });
          // 设置当前选项为活动状态
          this.classList.add('active');
        });
      });
      
      // 处理插入图片按钮
      sizingDialog.querySelector('.insert-image-btn').addEventListener('click', function() {
        const width = widthInput.value;
        const height = heightInput.value;
        const alignOption = sizingDialog.querySelector('.image-align-option.active');
        const align = alignOption ? alignOption.getAttribute('data-align') : 'none';
        
        // 构建图片HTML（使用HTML格式以支持样式）
        let styleAttr = `width: ${width}px; height: ${height}px;`;
        
        // 添加对齐样式
        if (align === 'left') {
          styleAttr += ' float: left; margin-right: 15px;';
        } else if (align === 'right') {
          styleAttr += ' float: right; margin-left: 15px;';
        } else if (align === 'center') {
          styleAttr += ' display: block; margin-left: auto; margin-right: auto;';
        }
        
        // 在光标位置插入图片HTML
        const imgHtml = `<img src="${imgPath}" alt="${fileName}" style="${styleAttr}">`;
        
        const start = textarea.selectionStart;
        const end = textarea.selectionEnd;
        textarea.value = textarea.value.substring(0, start) + imgHtml + textarea.value.substring(end);
        
        // 移除对话框
        sizingDialog.remove();
        
        // 添加成功消息
        showImageInsertSuccess(contentEditor, fileName);
        
        // 触发输入事件以更新预览
        textarea.dispatchEvent(new Event('input', { bubbles: true }));
        
        // 添加预览图片
        addImagePreview(contentEditor, e.target.result, fileName);
      });
      
      // 处理取消按钮
      sizingDialog.querySelector('.cancel-btn').addEventListener('click', function() {
        sizingDialog.remove();
      });
    };
  };
  reader.readAsDataURL(file);
}

// 显示图片插入成功消息
function showImageInsertSuccess(container, fileName) {
  let successMsg = container.querySelector('.image-insert-success');
  if (!successMsg) {
    successMsg = document.createElement('div');
    successMsg.className = 'image-insert-success';
    container.appendChild(successMsg);
  }
  successMsg.textContent = `Image '${fileName}' inserted, will be saved in img folder`;
  
  // 3秒后隐藏成功消息
  setTimeout(() => {
    successMsg.style.display = 'none';
  }, 3000);
}

// 添加图片预览
function addImagePreview(container, dataUrl, fileName) {
  const previewHolder = container.querySelector('.section-image-preview') || 
                      document.createElement('div');
  
  if (!previewHolder.classList.contains('section-image-preview')) {
    previewHolder.className = 'section-image-preview';
    previewHolder.style.marginTop = '10px';
    container.appendChild(previewHolder);
  }
  
  const previewImg = document.createElement('img');
  previewImg.className = 'img-preview';
  previewImg.src = dataUrl;
  previewImg.title = fileName;
  
  // 添加预览到容器
  previewHolder.innerHTML = '';
  previewHolder.appendChild(previewImg);
  
  // 添加预览标签
  const previewLabel = document.createElement('div');
  previewLabel.style.fontSize = '0.8em';
  previewLabel.style.marginTop = '5px';
  previewLabel.style.opacity = '0.7';
  previewLabel.textContent = `Preview: ${fileName}`;
  previewHolder.appendChild(previewLabel);
}