function generatePageTemplate(config) {
  // Validate required fields
  if (!config || !config.brandName || !config.pageTitle1) {
    throw new Error('Brand name and page title are required');
  }

  console.log('Generating page template with config:', config);

  // Get basic data
  const title = config.pageTitle1 || 'Untitled Page';
  const brand = config.brandName || 'My Brand';
  const slogan = config.brandSlogan || '';
  const themeStyle = config.themeStyle || 'light';
  const lightBgPath = config.lightBg || 'img/light-bg.jpg';
  const darkBgPath = config.darkBg || 'img/dark-bg.jpg';
  const logoPath = config.logoPath || 'img/logo.png';
  const faviconPath = config.faviconPath || 'favicon.ico';
  
  // Get column settings
  const gridColumns = config.gridColumns || 'auto';
  
  // Set column layout class
  let gridColumnsClass = '';
  if (gridColumns === '1') {
    gridColumnsClass = 'grid-one-column';
  } else if (gridColumns === '2') {
    gridColumnsClass = 'grid-two-columns';
  } else if (gridColumns === '3') {
    gridColumnsClass = 'grid-three-columns';
  } else {
    gridColumnsClass = 'grid-auto-columns';
  }
  
  // Build navigation menu
  const menuItemsHTML = Array.isArray(config.menuItems) 
    ? config.menuItems.map(item => 
        `<a href="${item.url || '#'}" class="nav-link">${item.text || 'Link'}</a>`
      ).join('')
    : '';
  
  // Build content sections - add custom column layout
  const sectionsHTML = Array.isArray(config.sections) && config.sections.length > 0
    ? `<div class="content-grid ${gridColumnsClass}">
        ${config.sections.map(section => {
          const sectionTitle = section.title || 'Untitled Section';
          // Parse Markdown content using marked
          let sectionContent = '';
          try {
            if (typeof marked !== 'undefined') {
              sectionContent = marked.parse(section.content || '');
            } else {
              sectionContent = section.content || '';
              console.error('Marked library not available');
            }
          } catch (e) {
            console.error('Markdown parsing error:', e);
            sectionContent = section.content || '';
          }
          
          // Content section card structure
          return `
            <div class="content-section card-style">
              <h2 class="section-title">${sectionTitle}</h2>
              <div class="section-body">${sectionContent}</div>
            </div>
          `;
        }).join('')}
      </div>`
    : '';
  
  // Build link sections
  const topLinksHTML = Array.isArray(config.topLinks)
    ? config.topLinks.map(link => 
        `<a href="${link.url || '#'}" class="top-link">${link.text || 'Link'}</a>`
      ).join('')
    : '';
  
  const bottomLinksHTML = Array.isArray(config.bottomLinks)
    ? config.bottomLinks.map(link => 
        `<a href="${link.url || '#'}" class="bottom-link">${link.text || 'Link'}</a>`
      ).join('')
    : '';
  
  const relatedLinksHTML = Array.isArray(config.relatedLinks)
    ? config.relatedLinks.map(link => 
        `<a href="${link.url || '#'}" class="related-link">${link.text || 'Link'}</a>`
      ).join('')
    : '';

  // Returns complete HTML with inline styles as CSS fallback
  return `
<!DOCTYPE html>
<html lang="en" data-theme="${themeStyle}">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>${title} - ${brand}</title>
  <meta name="description" content="${slogan}">
  
  <!-- Font references -->
  <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap">
  
  <!-- CSS file references - ensure correct file paths -->
  <link rel="stylesheet" href="css/main.css">
  <link rel="stylesheet" href="css/light.css">
  <link rel="stylesheet" href="css/dark.css">
  
  <!-- Favicon -->
  <link rel="icon" href="${faviconPath}" type="image/x-icon">
  
  <!-- Embedded critical styles for CSS fallback -->
  <style>
    /* Key variables and styles */
    :root[data-theme="light"] {
      --primary: #24283b;
      --secondary: #ffffff;
      --accent: #08FFC8;
      --accent-alt: #7B61FF;
      --bg-color: #f7f9fc;
      --card-bg: #ffffff;
      --text-color: #333333;
      --border-color: #dddddd;
      --light-bg-url: url('${lightBgPath}');
    }
    
    :root[data-theme="dark"] {
      --primary: #c8d3f5;
      --secondary: #1a1a2e;
      --accent: #08FFC8;
      --accent-alt: #7B61FF;
      --bg-color: #1a1a2e;
      --card-bg: #222244;
      --text-color: #f0f0f0;
      --border-color: #444466;
      --dark-bg-url: url('${darkBgPath}');
    }
    
    /* Base styles */
    body {
      font-family: 'Inter', sans-serif;
      margin: 0;
      padding: 0;
      color: var(--text-color);
      background-color: var(--bg-color);
      line-height: 1.6;
    }
    
    [data-theme="light"] body {
      background-image: var(--light-bg-url);
      background-size: cover;
      background-position: center;
      background-attachment: fixed;
    }
    
    [data-theme="dark"] body {
      background-image: var(--dark-bg-url);
      background-size: cover;
      background-position: center;
      background-attachment: fixed;
    }
    
    /* Navigation bar styles */
    .nav {
      background-color: var(--card-bg);
      box-shadow: 0 2px 10px rgba(0,0,0,0.1);
      position: sticky;
      top: 0;
      z-index: 100;
      width: 100%;
    }
    
    .nav-content {
      max-width: 1200px;
      margin: 0 auto;
      padding: 1rem 2rem;
      display: flex;
      justify-content: space-between;
      align-items: center;
    }
    
    .logo-container {
      display: flex;
      align-items: center;
      gap: 0.75rem;
    }
    
    .logo {
      height: 40px;
      width: auto;
    }
    
    .logo-text {
      font-weight: 700;
      font-size: 1.2rem;
      color: var(--primary);
    }
    
    .nav-links {
      display: flex;
      gap: 2rem;
    }
    
    .nav-links a {
      color: var(--text-color);
      text-decoration: none;
      font-weight: 600;
    }
    
    /* Container styles */
    .container {
      max-width: 1200px;
      margin: 2rem auto;
      padding: 0 2rem;
    }
    
    /* Content grid layout - column selection */
    .content-grid {
      display: grid;
      gap: 1.5rem;
      margin: 2rem 0;
    }
    
    /* Auto-adaptive column layout */
    .grid-auto-columns {
      grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
    }
    
    /* Fixed column layout */
    .grid-one-column {
      grid-template-columns: 1fr;
    }
    
    .grid-two-columns {
      grid-template-columns: repeat(2, 1fr);
    }
    
    .grid-three-columns {
      grid-template-columns: repeat(3, 1fr);
    }
    
    /* Mobile responsive handling */
    @media (max-width: 992px) {
      .grid-three-columns {
        grid-template-columns: repeat(2, 1fr);
      }
    }
    
    @media (max-width: 768px) {
      .content-grid,
      .grid-two-columns,
      .grid-three-columns {
        grid-template-columns: 1fr;
      }
      
      .nav-content {
        flex-direction: column;
        padding: 1rem;
      }
      
      .nav-links {
        margin-top: 1rem;
        gap: 1rem;
      }
      
      .container {
        padding: 0 1rem;
      }
    }
    
    /* Content section card styles */
    .content-section.card-style {
      background-color: var(--card-bg);
      border-radius: 12px;
      padding: 1.5rem;
      margin-bottom: 0;
      box-shadow: 0 4px 12px rgba(0,0,0,0.1);
      transition: all 0.3s ease;
      height: 100%;
      display: flex;
      flex-direction: column;
      border: 1px solid var(--border-color); /* Add border styling */
      overflow: hidden;
    }
    
    .content-section.card-style:hover {
      transform: translateY(-5px);
      box-shadow: 0 8px 24px rgba(0,0,0,0.15);
    }
    
    .content-section .section-title {
      margin-top: 0;
      margin-bottom: 1.2rem;
      font-weight: 700;
      font-size: 1.4rem;
      color: var(--primary);
      position: relative;
      padding-bottom: 0.8rem;
    }
    
    .content-section .section-title::after {
      content: '';
      position: absolute;
      bottom: 0;
      left: 0;
      width: 50px;
      height: 3px;
      background-color: var(--accent);
      border-radius: 1.5px;
    }
    
    .content-section .section-body {
      line-height: 1.6;
      flex: 1; /* Allow content area to fill remaining space */
    }
    
    .content-section .section-body img {
      max-width: 100%;
      border-radius: 8px;
    }
    
    [data-theme="dark"] .content-section.card-style {
      background-color: var(--card-bg);
      border: 1px solid var(--border-color);
    }
    
    [data-theme="dark"] .content-section .section-title {
      color: var(--text-primary);
    }
    
    [data-theme="dark"] .content-section .section-title::after {
      background-color: var(--accent);
    }
    
    /* Link container styles */
    .links-container {
      display: flex;
      flex-wrap: wrap;
      gap: 1rem;
      margin: 1.5rem 0;
    }
    
    .links-container a {
      color: var(--accent);
      text-decoration: none;
      padding: 0.5rem 1rem;
      border: 1px solid var(--border-color);
      border-radius: 8px;
      transition: all 0.3s ease;
    }
    
    .links-container a:hover {
      background-color: var(--accent);
      color: var(--bg-color);
    }
    
    /* Footer styles */
    footer {
      text-align: center;
      padding: 2rem;
      margin-top: 3rem;
      background-color: var(--card-bg);
      color: var(--text-color);
      border-top: 1px solid var(--border-color);
    }
    
    /* General styles */
    h1, h2, h3 {
      color: var(--primary);
    }
    
    img {
      max-width: 100%;
      height: auto;
    }
  </style>
  
  <!-- Ensure Markdown parser library is available -->
  <script src="https://cdn.jsdelivr.net/npm/marked@4.0.0/marked.min.js"></script>
</head>
<body>
  <div class="nav">
    <div class="nav-content">
      <div class="logo-container">
        <img src="${logoPath}" class="logo" alt="${brand}">
        <span class="logo-text">${brand}</span>
      </div>
      <div class="nav-links">
        ${menuItemsHTML}
      </div>
    </div>
  </div>
  
  <div class="container">
    <h1>${title}</h1>
    ${slogan ? `<p class="brand-slogan">${slogan}</p>` : ''}
    
    <div class="links-container">
      ${topLinksHTML}
    </div>
    
    ${sectionsHTML}
    
    <div class="links-container">
      ${bottomLinksHTML}
    </div>
    
    <div class="links-container">
      <h3>more...</h3>
      → 
      <h3>Related Links</h3>
      ${relatedLinksHTML}
    </div>
  </div>
  
  <footer>
    <p>© ${config.copyrightYear || new Date().getFullYear()} ${config.companyName || brand}. All rights reserved.</p>
  </footer>

  <!-- Theme initialization script -->
  <script>
    document.addEventListener('DOMContentLoaded', function() {
      // Ensure theme styles are applied
      document.documentElement.setAttribute('data-theme', '${themeStyle}');
      
      // Simple theme toggle functionality
      function toggleTheme() {
        const currentTheme = document.documentElement.getAttribute('data-theme');
        const newTheme = currentTheme === 'light' ? 'dark' : 'light';
        document.documentElement.setAttribute('data-theme', newTheme);
        localStorage.setItem('theme', newTheme);
      }
      
      // Check if theme toggle button exists
      const themeToggle = document.getElementById('themeToggle');
      if (themeToggle) {
        themeToggle.addEventListener('click', toggleTheme);
      }
    });
  </script>
</body>
</html>
  `;
}